<?php
/**
 * Plugin Name: WhatsApp Notifier
 * Description: Sends WhatsApp messages to customers after any WooCommerce order status change.
 * Version: 2.0
 * Author: YourName
 */

if (!defined('ABSPATH')) exit;

// Settings page
add_filter('woocommerce_get_settings_pages', function($settings) {
    $settings[] = new class extends WC_Settings_Page {
        public function __construct() {
            $this->id = 'whatsapp_notifier';
            $this->label = 'WhatsApp Notifier';
            parent::__construct();
        }

        public function get_settings() {
            return apply_filters('woocommerce_settings_tabs_whatsapp_notifier', [
                ['title' => 'WhatsApp Notifier Settings', 'type' => 'title', 'id' => 'wn_title'],
                [
                    'title' => 'Enable WhatsApp Notifications',
                    'id' => 'wn_enable',
                    'type' => 'checkbox',
                    'default' => 'no',
                ],
                [
                    'title' => 'Instance ID',
                    'id' => 'wn_instance_id',
                    'type' => 'text',
                ],
                [
                    'title' => 'Access Token',
                    'id' => 'wn_access_token',
                    'type' => 'text',
                ],
                [
                    'title' => 'Custom Message Template',
                    'id' => 'wn_message_template',
                    'type' => 'textarea',
                    'desc' => 'Use shortcodes: {first_name}, {last_name}, {order_id}, {items}, {shipping}, {total}, {status}',
                    'default' => 'Hi {first_name} {last_name}, your order #{order_id} is now {status}.\nItems:\n{items}\nShipping: {shipping}\nTotal: {total}',
                ],
                ['type' => 'sectionend', 'id' => 'wn_title'],
            ]);
        }
    };
    return $settings;
});

// Send message on ANY status change
add_action('woocommerce_order_status_changed', 'wn_send_whatsapp_message_on_status_change', 10, 4);
function wn_send_whatsapp_message_on_status_change($order_id, $old_status, $new_status, $order) {
    if ('yes' !== get_option('wn_enable')) return;

    $instance_id = get_option('wn_instance_id');
    $access_token = get_option('wn_access_token');
    $template = get_option('wn_message_template');

    $first_name = $order->get_billing_first_name();
    $last_name = $order->get_billing_last_name();
    $phone = normalize_phone_number($order->get_billing_phone());
    $order_id_text = $order->get_order_number();

    $items = '';
    foreach ($order->get_items() as $item) {
        $items .= $item->get_name() . ' x ' . $item->get_quantity() . ' = ₹' . number_format($item->get_total(), 2) . "\n";
    }

    $shipping_total = 0;
    foreach ($order->get_items('shipping') as $shipping) {
        $shipping_total += $shipping->get_total();
    }
    $shipping_text = $shipping_total > 0 ? '₹' . number_format($shipping_total, 2) : '₹0.00';
    $total = '₹' . number_format($order->get_total(), 2);
    $status = ucfirst($new_status);

    $message = str_replace(
        ['{first_name}', '{last_name}', '{order_id}', '{items}', '{shipping}', '{total}', '{status}'],
        [$first_name, $last_name, $order_id_text, $items, $shipping_text, $total, $status],
        $template
    );

    $url = 'https://cloud.connectsdigital.in/api/send?number=' . $phone . '&type=text&message=' . urlencode($message) . '&instance_id=' . $instance_id . '&access_token=' . $access_token;
    wp_remote_get($url);
}

// Normalize phone number to international format
function normalize_phone_number($phone) {
    $phone = preg_replace('/\D/', '', $phone);
    if (substr($phone, 0, 2) !== '91') {
        $phone = '91' . $phone;
    }
    return $phone;
}
